using System;
using System.Data;
using System.Data.SqlClient;

using gov.va.med.vbecs.Common;
using Sprocs = gov.va.med.vbecs.Common.VbecsStoredProcs;
using VistALinkControlTable = gov.va.med.vbecs.Common.VbecsTables.VistaLinkControl;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/12/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// Implements data persisting logic for accesing VistALink configuration in the VBECS DB.
	/// Provides methods to retrieve and update VistALink configration.
	/// </summary>

	#endregion

	public class VistALinkConfig
	{
		private VistALinkConfig() {}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5568"> 
		///		<ExpectedInput>None. VistALink configuration is already defined in the DB.</ExpectedInput>
		///		<ExpectedOutput>Single DataRow containing VistALink configuration data.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5589"> 
		///		<ExpectedInput>None. VistALink configuration is not defined in the DB.</ExpectedInput>
		///		<ExpectedOutput>Null.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5569"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads VistALink configuration from DB. If there's no VistALink config defined - will return null.
		/// </summary>
		/// <returns><see cref="DataRow"/> containing VistALink configuration data.</returns>
		public static DataRow Load( string serverAppHandle )
		{
			if( serverAppHandle == null )
				throw( new ArgumentNullException( "serverAppHandle" ) );

			SqlParameter _param = new SqlParameter( Sprocs.GetVistALinkConfig.serverapphandle, SqlDbType.VarChar );
			_param.Value = serverAppHandle;

			DataTable _dt = StoredProcedure.GetDataTable( Sprocs.GetVistALinkConfig.StoredProcName, _param );

			// VistALink config isn't defined in the DB. 
			if( _dt.Rows.Count == 0 )
				return null;

			if( _dt.Rows.Count > 1 ) 
				throw( new DataAccessLayerException( StrRes.SysErrMsg.Common.InsufficientNumberOfDataRowsFoundWhileExpectingSingleRow( Sprocs.GetVistALinkConfig.StoredProcName, _dt.Rows.Count ).ResString ) );

			return _dt.Rows[0];
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5570"> 
		///		<ExpectedInput>Valid DataRow with test values, DB already contains VistALink config.</ExpectedInput>
		///		<ExpectedOutput>None, values from DataRow are saved in the DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5590"> 
		///		<ExpectedInput>Valid DataRow with test values, DB does not contain VistALink config (fresh install).</ExpectedInput>
		///		<ExpectedOutput>None, values from DataRow are saved in the DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5571"> 
		///		<ExpectedInput>Null instead of DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5574"> 
		///		<ExpectedInput>DataRow with no columns</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates existing VistALink configuration record in the DB or inserts new configuration record 
		/// (there can be only one VistALink configuration record in the database).
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to save data from.</param>
		public static void Save( DataRow dtRow )
		{			
			Utility.RequireNonNullColumns( dtRow, VistALinkControlTable.ServerAppHandle, VistALinkControlTable.ServerAlias, VistALinkControlTable.VistaPortNumber, VistALinkControlTable.LastUpdateFunctionId );
			Utility.RequireColumns( dtRow, VistALinkControlTable.RowVersion ); // there may be no rowversion right after the installation.
			//
			// If RowVersion is NULL, then INSERT, else UPDATE
			if( dtRow.IsNull( VistALinkControlTable.RowVersion ) )
			{
				dtRow.Table.Columns.Remove( VistALinkControlTable.RowVersion );
				new StoredProcedure().TransactionalGetValue( "spInsertVistALinkConfig", dtRow );
			}
			else
			{
				new StoredProcedure().TransactionalGetValue( Sprocs.UpdateVistALinkConfig.StoredProcName, dtRow );
			}	
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5572"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5573"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a new instance of <see cref="DataTable"/> used 
		/// for VistALink config related data exchange.
		/// </summary>
		/// <returns>Template <see cref="DataTable"/> used for VistALinkConfig data exchange.</returns>
		public static DataTable GetVistaLinkControlEmptyDataTable()
		{
			DataTable _dt = new DataTable();

			_dt.Columns.Add( VistALinkControlTable.ServerAppHandle, typeof(string) );
			_dt.Columns.Add( VistALinkControlTable.ServerAlias, typeof(string) );
			_dt.Columns.Add( VistALinkControlTable.VistaIpAddress, typeof(string) );
			_dt.Columns.Add( VistALinkControlTable.VistaDomainName, typeof(string) );
			_dt.Columns.Add( VistALinkControlTable.VistaPortNumber, typeof(int) );
            _dt.Columns.Add(VistALinkControlTable.VbecsServiceIpAddress, typeof(string)); // CR 3419
            _dt.Columns.Add(VistALinkControlTable.VbecsServicePortNumber, typeof(int)); // CR 3419
			_dt.Columns.Add( VistALinkControlTable.RowVersion, typeof(byte[]) );
			_dt.Columns.Add( VistALinkControlTable.LastUpdateFunctionId, typeof(int) );

			return _dt;
		}
	}
}
